<?php

defined('BASEPATH') or exit('No direct script access allowed');

class Job_daily_logs_model extends App_Model {

	/**
	 * Get daily log by id
	 *
	 * @param  int  $id  daily log id
	 *
	 * @return object|null
	 */
	public function get($id)
	{
		$this->db->select(
			db_prefix().'tjm_job_daily_logs.*, '.
			'CONCAT('.db_prefix().'staff.firstname, " ", '.db_prefix().'staff.lastname) as staff_full_name, '.
			db_prefix().'staff.firstname, '.
			db_prefix().'staff.lastname',
		);
		$this->db->from(db_prefix().'tjm_job_daily_logs');
		$this->db->join(db_prefix().'staff', db_prefix().'staff.staffid = '.db_prefix().'tjm_job_daily_logs.staff_id', 'left');
		$this->db->where(db_prefix().'tjm_job_daily_logs.daily_log_id', $id);

		$log = $this->db->get()->row();

		if ($log)
		{
			$log->photos = $this->get_photos($id);
		}

		return $log;
	}

	/**
	 * Get all daily logs for a job
	 *
	 * @param  int  $job_id  job id
	 * @param  bool  $customer_view  if true, only return visible logs
	 *
	 * @return array
	 */
	public function get_by_job_id($job_id, bool $customer_view = FALSE)
	{
		$this->db->select(
			db_prefix().'tjm_job_daily_logs.*, '.
			'CONCAT('.db_prefix().'staff.firstname, " ", '.db_prefix().'staff.lastname) as staff_full_name, '.
			db_prefix().'staff.firstname, '.
			db_prefix().'staff.lastname',
		);
		$this->db->from(db_prefix().'tjm_job_daily_logs');
		$this->db->join(db_prefix().'staff', db_prefix().'staff.staffid = '.db_prefix().'tjm_job_daily_logs.staff_id', 'left');
		$this->db->where(db_prefix().'tjm_job_daily_logs.job_id', $job_id);

		if ($customer_view)
		{
			$this->db->where(db_prefix().'tjm_job_daily_logs.is_visible_to_customer', 1);
		}

		$this->db->order_by(db_prefix().'tjm_job_daily_logs.log_date', 'DESC');
		$this->db->order_by(db_prefix().'tjm_job_daily_logs.created_at', 'DESC');

		$logs = $this->db->get()->result_array();

		// Attach photos to each log
		foreach ($logs as $key => $log)
		{
			$logs[$key]['photos'] = $this->get_photos($log['daily_log_id']);
		}

		return $logs;
	}

	/**
	 * Get daily logs by specific date
	 *
	 * @param  int  $job_id  job id
	 * @param  string  $date  date in Y-m-d format
	 * @param  bool  $customer_view  if true, only return visible logs
	 *
	 * @return array
	 */
	public function get_by_date($job_id, $date, $customer_view = FALSE)
	{
		$this->db->select(
			db_prefix().'tjm_job_daily_logs.*, '.
			'CONCAT('.db_prefix().'staff.firstname, " ", '.db_prefix().'staff.lastname) as staff_full_name, '.
			db_prefix().'staff.firstname, '.
			db_prefix().'staff.lastname',
		);
		$this->db->from(db_prefix().'tjm_job_daily_logs');
		$this->db->join(db_prefix().'staff', db_prefix().'staff.staffid = '.db_prefix().'tjm_job_daily_logs.staff_id', 'left');
		$this->db->where(db_prefix().'tjm_job_daily_logs.job_id', $job_id);
		$this->db->where(db_prefix().'tjm_job_daily_logs.log_date', $date);

		if ($customer_view)
		{
			$this->db->where(db_prefix().'tjm_job_daily_logs.is_visible_to_customer', 1);
		}

		$this->db->order_by(db_prefix().'tjm_job_daily_logs.created_at', 'DESC');

		$logs = $this->db->get()->result_array();

		// Attach photos to each log
		foreach ($logs as $key => $log)
		{
			$logs[$key]['photos'] = $this->get_photos($log['daily_log_id']);
		}

		return $logs;
	}

	/**
	 * Get dates with logs for a job (for calendar display)
	 *
	 * @param  int  $job_id  job id
	 * @param  bool  $customer_view  if true, only return visible logs
	 *
	 * @return array
	 */
	public function get_dates_with_logs($job_id, $customer_view = FALSE)
	{
		$this->db->select('DISTINCT log_date');
		$this->db->from(db_prefix().'tjm_job_daily_logs');
		$this->db->where('job_id', $job_id);

		if ($customer_view)
		{
			$this->db->where('is_visible_to_customer', 1);
		}

		$this->db->order_by('log_date', 'DESC');

		$result = $this->db->get()->result_array();

		return array_column($result, 'log_date');
	}

	/**
	 * Add new daily log
	 *
	 * @param  array  $data  log data
	 *
	 * @return int|bool
	 */
	public function add($data)
	{
		$insert_data = [
			'job_id'                 => $data['job_id'],
			'log_date'               => $data['log_date'] ?? date('Y-m-d'),
			'comments'               => $data['comments'] ?? '',
			'is_visible_to_customer' => $data['is_visible_to_customer'] ?? 1,
			'staff_id'               => $data['staff_id'] ?? get_staff_user_id(),
			'created_at'             => date('Y-m-d H:i:s'),
			'updated_at'             => date('Y-m-d H:i:s'),
		];

		hooks()->do_action('before_add_job_daily_log', $insert_data);

		$this->db->insert(db_prefix().'tjm_job_daily_logs', $insert_data);
		$insert_id = $this->db->insert_id();

		if ($insert_id)
		{
			log_activity('New Job Daily Log Added [Job ID: '.$data['job_id'].', Date: '.$data['log_date'].']');
			hooks()->do_action('after_add_job_daily_log', $insert_id);

			return $insert_id;
		}

		return FALSE;
	}

	/**
	 * Update daily log
	 *
	 * @param  int  $id  daily log id
	 * @param  array  $data  log data
	 *
	 * @return bool
	 */
	public function update($id, $data)
	{
		$update_data = [
			'log_date'               => $data['log_date'] ?? date('Y-m-d'),
			'comments'               => $data['comments'] ?? '',
			'is_visible_to_customer' => $data['is_visible_to_customer'] ?? 1,
			'updated_at'             => date('Y-m-d H:i:s'),
		];

		hooks()->do_action('before_update_job_daily_log', ['id' => $id, 'data' => $update_data]);

		$this->db->where('daily_log_id', $id);
		$this->db->update(db_prefix().'tjm_job_daily_logs', $update_data);

		if ($this->db->affected_rows() > 0)
		{
			hooks()->do_action('after_update_job_daily_log', $id);

			return TRUE;
		}

		return FALSE;
	}

	/**
	 * Delete daily log and all associated photos
	 *
	 * @param  int  $id  daily log id
	 *
	 * @return bool
	 */
	public function delete($id)
	{
		$log = $this->get($id);

		if ( ! $log)
		{
			return FALSE;
		}

		hooks()->do_action('before_delete_job_daily_log', $id);

		// Delete associated photos
		$photos = $this->get_photos($id);
		foreach ($photos as $photo)
		{
			$this->delete_photo($log->job_id, $photo['id']);
		}

		$this->db->where('daily_log_id', $id);
		$this->db->delete(db_prefix().'tjm_job_daily_logs');

		if ($this->db->affected_rows() > 0)
		{
			log_activity('Job Daily Log Deleted [ID: '.$id.']');
			hooks()->do_action('after_delete_job_daily_log', $id);

			return TRUE;
		}

		return FALSE;
	}

	/**
	 * Get all photos for a daily log
	 *
	 * @param  int  $daily_log_id  daily log id
	 *
	 * @return array
	 */
	public function get_photos($daily_log_id)
	{
		$this->db->where('rel_id', $daily_log_id);
		$this->db->where('rel_type', 'job_daily_log');
		$this->db->order_by('dateadded', 'ASC');

		return $this->db->get(db_prefix().'files')->result_array();
	}

	/**
	 * Get daily log by photo id
	 *
	 * @param  int  $id  photo id
	 *
	 * @return object|null
	 */
	public function get_by_photo_id($id)
	{
		$this->db->select(db_prefix().'tjm_job_daily_logs.* ');
		$this->db->from(db_prefix().'tjm_job_daily_logs');
		$this->db->join(db_prefix().'files', db_prefix().'files.rel_id = '.db_prefix().'tjm_job_daily_logs.daily_log_id');
		$this->db->where(db_prefix().'files.id', $id);
		$this->db->where(db_prefix().'files.rel_type', 'job_daily_log');

		return $this->db->get()->row();
	}

	/**
	 * Delete a photo
	 *
	 * @param  int  $file_id  file id
	 *
	 * @return bool
	 */
	public function delete_photo($job_id, $file_id)
	{
		$this->db->where('id', $file_id);
		$attachment = $this->db->get(db_prefix().'files')->row();

		if ($attachment && $attachment->rel_type === 'job_daily_log')
		{
			$this->db->where('id', $file_id);
			$this->db->delete(db_prefix().'files');

			$relPath  = get_upload_path_by_type('job').$job_id.'/daily_logs/';
			$fullPath = $relPath.$attachment->file_name;
			if (file_exists($fullPath))
			{
				unlink($fullPath);
				$fname     = pathinfo($fullPath, PATHINFO_FILENAME);
				$fext      = pathinfo($fullPath, PATHINFO_EXTENSION);
				$thumbPath = $relPath.$fname.'_thumb.'.$fext;
				if (file_exists($thumbPath))
				{
					unlink($thumbPath);
				}
			}

			return TRUE;
		}

		return FALSE;
	}

	/**
	 * Check if staff can edit a daily log (only today's logs)
	 *
	 * @param  int  $daily_log_id  daily log id
	 * @param  int  $staff_id  staff id
	 *
	 * @return bool
	 */
	public function can_edit($daily_log_id, $staff_id)
	{
		// Admins can edit any log
		if (is_admin())
		{
			return TRUE;
		}

		$this->db->where('daily_log_id', $daily_log_id);
		$this->db->where('staff_id', $staff_id);
		$log = $this->db->get(db_prefix().'tjm_job_daily_logs')->row();

		if ( ! $log)
		{
			return FALSE;
		}

		// Can only edit today's logs
		return $log->log_date === date('Y-m-d');
	}

	/**
	 * Check if staff can delete a daily log (only today's logs)
	 *
	 * @param  int  $daily_log_id  daily log id
	 * @param  int  $staff_id  staff id
	 *
	 * @return bool
	 */
	public function can_delete($daily_log_id, $staff_id)
	{
		// Admins can delete any log
		if (is_admin())
		{
			return TRUE;
		}

		$this->db->where('daily_log_id', $daily_log_id);
		$this->db->where('staff_id', $staff_id);
		$log = $this->db->get(db_prefix().'tjm_job_daily_logs')->row();

		if ( ! $log)
		{
			return FALSE;
		}

		// Can only delete today's logs
		return $log->log_date === date('Y-m-d');
	}

	/**
	 * Get summary statistics for a job's daily logs
	 *
	 * @param  int  $job_id  job id
	 *
	 * @return array
	 */
	public function get_statistics($job_id)
	{
		$this->db->select('COUNT(*) as total_logs, COUNT(DISTINCT log_date) as total_days');
		$this->db->where('job_id', $job_id);
		$result = $this->db->get(db_prefix().'tjm_job_daily_logs')->row_array();

		// Count total photos
		$this->db->select('COUNT(*) as total_photos');
		$this->db->from(db_prefix().'files');
		$this->db->join(db_prefix().'tjm_job_daily_logs', db_prefix().'files.rel_id = '.db_prefix().'tjm_job_daily_logs.daily_log_id');
		$this->db->where(db_prefix().'files.rel_type', 'job_daily_log');
		$this->db->where(db_prefix().'tjm_job_daily_logs.job_id', $job_id);
		$photo_result = $this->db->get()->row_array();

		return [
			'total_logs'   => (int)$result['total_logs'],
			'total_days'   => (int)$result['total_days'],
			'total_photos' => (int)$photo_result['total_photos'],
		];
	}

}
